<?php
/**
 * STELLAZODIACO™ - TEST MINIMALE per DEBUG
 * Questo file testa solo la connessione base senza Stripe
 */

// ✅ Headers per CORS e JSON
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// ✅ Gestione preflight OPTIONS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// ✅ Error reporting
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', 'test_errors.log');

// ✅ Logging funzione
function logMessage($message) {
    error_log(date('[Y-m-d H:i:s] ') . $message);
}

try {
    logMessage("=== TEST STELLAZODIACO START ===");
    logMessage("Method: " . $_SERVER['REQUEST_METHOD']);
    logMessage("Content-Type: " . ($_SERVER['CONTENT_TYPE'] ?? 'not set'));
    
    // ✅ Verifica metodo POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Solo richieste POST sono accettate');
    }
    
    // ✅ Leggi input
    $input = file_get_contents('php://input');
    logMessage("Raw input: " . substr($input, 0, 200));
    
    if (empty($input)) {
        // Prova $_POST se input vuoto
        if (!empty($_POST)) {
            $data = $_POST;
            logMessage("Using \$_POST data");
        } else {
            throw new Exception('Nessun dato ricevuto');
        }
    } else {
        // Decodifica JSON
        $data = json_decode($input, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('JSON non valido: ' . json_last_error_msg());
        }
        logMessage("JSON decoded successfully");
    }
    
    // ✅ Verifica azione
    $action = $data['action'] ?? '';
    logMessage("Action: " . $action);
    
    if ($action === 'test_connection') {
        // ✅ Test connessione base
        logMessage("Processing test_connection");
        
        // Valida campi richiesti
        $requiredFields = ['firstName', 'lastName', 'email', 'phone'];
        $missingFields = [];
        
        foreach ($requiredFields as $field) {
            if (empty($data[$field])) {
                $missingFields[] = $field;
            }
        }
        
        if (!empty($missingFields)) {
            throw new Exception('Campi mancanti: ' . implode(', ', $missingFields));
        }
        
        // ✅ Simula elaborazione
        logMessage("All fields validated");
        
        // Test email valida
        if (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            throw new Exception('Email non valida');
        }
        
        logMessage("Email validation passed");
        
        // ✅ Risposta di successo
        $response = [
            'success' => true,
            'message' => 'Test connessione completato con successo!',
            'data' => [
                'nome_completo' => $data['firstName'] . ' ' . $data['lastName'],
                'email' => $data['email'],
                'timestamp' => date('Y-m-d H:i:s'),
                'test_id' => 'TEST_' . time()
            ],
            'php_info' => [
                'version' => PHP_VERSION,
                'server' => $_SERVER['SERVER_SOFTWARE'] ?? 'unknown',
                'memory_usage' => memory_get_usage(true),
                'time' => microtime(true) - $_SERVER['REQUEST_TIME_FLOAT']
            ]
        ];
        
        logMessage("Success response prepared");
        
        echo json_encode($response, JSON_PRETTY_PRINT);
        
    } elseif ($action === 'create_stripe_session') {
        // ✅ Test Stripe (senza libreria)
        logMessage("Processing create_stripe_session (simulated)");
        
        // Simula successo Stripe
        $response = [
            'success' => true,
            'session_id' => 'cs_test_' . time() . '_' . rand(1000, 9999),
            'message' => 'Sessione Stripe simulata creata con successo',
            'note' => 'Questa è una simulazione - Stripe non ancora configurato'
        ];
        
        echo json_encode($response);
        
    } else {
        throw new Exception('Azione non riconosciuta: ' . $action);
    }
    
    logMessage("=== TEST STELLAZODIACO SUCCESS ===");
    
} catch (Exception $e) {
    logMessage("ERROR: " . $e->getMessage());
    
    $response = [
        'success' => false,
        'message' => $e->getMessage(),
        'error_details' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ],
        'debug_info' => [
            'php_version' => PHP_VERSION,
            'timestamp' => date('Y-m-d H:i:s'),
            'memory_usage' => memory_get_usage(true)
        ]
    ];
    
    http_response_code(400);
    echo json_encode($response, JSON_PRETTY_PRINT);
    
    logMessage("=== TEST STELLAZODIACO ERROR ===");
}
?>